# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2008-2009 Gerd Neugebauer gene@gerd-neugebauer.de
# All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the Library GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#

=pod

---+ package LaTeXSyntaxPlugin

This is a TWiki plugin to render LaTeX syntax

=cut

# change the package name and $pluginName!!!
package TWiki::Plugins::LaTeXSyntaxPlugin;

# Always use strict to enforce variable scoping
use strict;

# $VERSION is referred to by TWiki, and is the only global variable that
# *must* exist in this package.
use vars qw( $VERSION
	     $RELEASE
	     $SHORTDESCRIPTION
	     $debug
	     $pluginName
	     $NO_PREFS_IN_TOPIC
	     %LaTeXSyntax);

# This should always be $Rev: 12445$ so that TWiki can determine the checked-in
# status of the plugin. It is used by the build automation tools, so
# you should leave it alone.
$VERSION = '$Rev: 12445$'; #'

# This is a free-form string you can use to "name" your own plugin version.
# It is *not* used by the build automation tools, but is reported as part
# of the version number in PLUGINDESCRIPTIONS.
$RELEASE = 'Dakar';

$SHORTDESCRIPTION = 'LaTeX Syntax Plugin rendering LaTeX syntax to HTML';

# You must set $NO_PREFS_IN_TOPIC to 0 if you want your plugin to use preferences
# stored in the plugin topic. This default is required for compatibility with
# older plugins, but imposes a significant performance penalty, and
# is not recommended. Instead, use $TWiki::cfg entries set in LocalSite.cfg, or
# if you want the users to be able to change settings, then use standard TWiki
# preferences that can be defined in your Main.TWikiPreferences and overridden
# at the web and topic level.
$NO_PREFS_IN_TOPIC = 1;

# Name of this Plugin, only used in this module
$pluginName = 'LaTeXSyntaxPlugin';

use FileHandle;

use constant FOOTNOTE_COUNT => 9;
use constant TABULAR_COUNT => 8;
use constant TABULAR_FORMAT => 7;
use constant FLOAT_TYPE => 6;
use constant STATE_STACK => 5;
use constant STATE => 4;
# state values
use constant STATE_TEXT => 1;
use constant STATE_MATH => 2;
use constant STATE_ITEMIZE => 3;
use constant STATE_ENUMERATE => 4;
use constant STATE_DESCRIPTION_1 => 5;
use constant STATE_DESCRIPTION => 6;

=pod

---++ initPlugin($topic, $web, $user, $installWeb) -> $boolean
   * =$topic= - the name of the topic in the current CGI query
   * =$web= - the name of the web in the current CGI query
   * =$user= - the login name of the user
   * =$installWeb= - the name of the web the plugin is installed in

REQUIRED

Called to initialise the plugin.

=cut

sub initPlugin {
  my( $topic, $web, $user, $installWeb ) = @_;

  if( $TWiki::Plugins::VERSION < 1.026 ) {
    TWiki::Func::writeWarning("Version mismatch between $pluginName and Plugins.pm");
    return 0;
  }
  $debug   = $TWiki::cfg{Plugins}{LaTeXSyntaxPlugin}{Debug} || 0;

#  TWiki::Func::registerTagHandler( 'LATEXSYNTAX', \&_LATEXSYNTAX );

  return 1;
}

sub _LATEXSYNTAX {
    my($session, $params, $theTopic, $theWeb) = @_;
    # $session  - a reference to the TWiki session object (if you don't know
    #             what this is, just ignore it)
    # $params=  - a reference to a TWiki::Attrs object containing parameters.
    #             This can be used as a simple hash that maps parameter names
    #             to values, with _DEFAULT being the name for the default
    #             parameter.
    # $theTopic - name of the topic in the query
    # $theWeb   - name of the web in the query
    # Return: the result of processing the variable

    # For example, %EXAMPLETAG{'hamburger' sideorder="onions"}%
    # $params->{_DEFAULT} will be 'hamburger'
    # $params->{sideorder} will be 'onions'
}

=pod

---++ scan_opt($input) -> $value

Take the input string and extract an optional argument enclosed in
brackets from its start. The contents of the argument is returned. If
the input does not start with a bracket then undef is returned.

=cut

sub scan_opt {
  $_[0] =~ s/^[ \t\r\n]*//;
  if (length($_[0]) == 0) {
    TWiki::Func::writeWarning("unexpected eof" );
    return undef ;
  }
  local $_ = substr($_[0],0,1);
  if ($_ ne '[') {
    return undef;
  }

  $_[0] = substr($_[0],1);

  if ($_[0] =~ m/\]/) {
    $_[0] = $';
    return $`;
  }
  # TODO error handling and counting of brace levels
  die "unimplemented"
}

sub scan_star {
  if ($_[0] =~ s/^[ \t\r\n]*\\*//) {
    $_[0] = $';
    return 1;
  }
  return undef;
}

=pod

---++ scan_arg($input) -> $value

Take the input string and extract an argument enclosed in
braces from its start. The contents of the argument is returned. If
the input does not start with a brace then the first control sequence
or character found is returned.

=cut

sub scan_arg {
  $_[0]		=~ s/^[ \t\r\n]*//;
  if (not $_[0] =~ m/^./) {
    TWiki::Func::writeWarning("unexpected eof");
    return undef;
  }
  local $_ = $&;
  $_[0]    = $';
  if ($_ eq '\\') {
    if ($_[0] =~ m/^[a-zA-Z]+|./) {
      $_   .= $&;
      $_[0] = $';
    }
    return $_;
  } elsif ($_ ne '{') {
    return $_;
  }
  my $depth  = 1;
  my $result = '';
  while($_[0] =~ m/[{}\\]/) {
    $result  .= $`;
    $_[0]     = $';
    if ($& eq '{') {
      $depth++;
      $result .= $&;
      $_[0] .= substr($_[0],1);
    } elsif ($& eq '}') {
      $depth--;
      if ($depth == 0) {
	return $result;
      }
      $result .= $&;
      $_[0] .= substr($_[0],1);
    } elsif ($& eq '\\') {
      $result .= substr($_[0],0,2);
      $_[0] .= substr($_[0],2);
    } else {
      $result .= $&;
      $_[0] .= substr($_[0],1);
    }
  }
  TWiki::Func::writeWarning("unmatched parenthesis");
  return undef;
}

sub expanded_arg {
  local $_ = scan_arg(@_);
  my $save = $_[0];
  $_[0]    = $_;
  $_	   = expand_LaTeX(@_);
  $_[0]    = $save;
  return $_;
}

=pod

---++ macro_includegraphics($input) -> $string

Code for \includegraphics

=cut

sub macro_includegraphics {
  my $opt = scan_opt(@_) || '';
  my $arg = scan_arg(@_);
  local $_;
  my $a = '';

  TWiki::Func::writeWarning("--- includegraphics $opt") if $debug;

  if ($opt =~ m/width=([\\0-9.a-z]*)/) {
    $_ = $1;
    if ( s/\\(textwidth|linelength|pagewidth)//) {
      $_ = $` * 100;
      $a .= " width=\"$_%\"";
    } else {
      s/[^0-9]//g;
      $_ *= 10;
      $a .= " width=\"$_\"";
    }
    TWiki::Func::writeWarning("--- width = $_") if $debug;
  }
  if ($opt =~ m/height=([\\0-9.a-z]*)/) {
    $_ = $1;
    if ( s/\\(pageheight)//) {
      $_ = $` * 100;
      $a .= " height=\"$_%\"";
    } else {
      s/[^0-9]//g;
      $_ *= 10;
      $a .= " height=\"$_\"";
    }
  }
  return "<img src=\"$arg\"$a/>";
}

=pod

---++ macro_verb($input) -> $string

Code for \verb

=cut

sub macro_verb {
  if (not ($_[0] =~ m/^[ ]*(.)/)) {
    return '\\verb';
  }
  $_[0]    = $';
  local $_ = index($_[0], $1);
  if ($_ < 0) {
    TWiki::Func::writeWarning("\verb end not found");
    return '\\verb' . $';
  }
  my $s = substr($_[0], 0, $_);
  $_[0] = substr($_[0], $_ + 1);
  return "<code>$s</code>";
}

=pod

---++ env_verbatim($input) -> $string

Code for the verbatim environment.

=cut

sub env_verbatim {
  local $_	= index('\\end{verbatim}',$_[0]);
  if (not $_[0] =~ m/\\end{verbatim}/) {
    TWiki::Func::writeWarning("verbatim end not found");
    return '\\begin{verbatim}';
  }
  $_[0] = $';
  return "<pre>$`</pre>";
}

sub env_lstlisting {
  scan_opt(@_); # gobble optional argument
  local $_	= index('\\end{lstlisting}',$_[0]);
  if (not $_[0] =~ m/\\end{lstlisting}/) {
    TWiki::Func::writeWarning("lstlisting end not found");
    return '\\begin{lstlisting}';
  }
  $_[0] = $';
  return "<pre>$`</pre>";
}

sub include {
  local $_ = scan_arg(@_);
  if (not defined $_) {
    return '';
  }
  my $fd = new FileHandle($_);
  my $s  = '';
  while (<$fd>) {
    $s .= $_ if not m/^[ \t]%/;
  }
  $fd->close();
  $_[0] = $s . $_[0];
  return '';
}

sub begin_math {
  push @{$_[STATE_STACK]}, $_[STATE];
  $_[STATE] = STATE_MATH;
  return '<i>';
}

sub end_math {
  $_[STATE] = pop @{$_[STATE_STACK]};
  return '</i>';
}

sub begin_displaymath {
  push @{$_[STATE_STACK]}, $_[STATE];
  $_[STATE] = STATE_MATH;
  return '<p><i>';
}

sub end_displaymath {
  $_[STATE] = pop @{$_[STATE_STACK]};
  return '</i></p>';
}

sub parseColSpec {
  return [];
}

sub begin_tabular {
#  my @cols 	 = parseColSpec(scan_opt{@_});
  my $fmt	 = scan_arg(@_);
  my $toprule	 = undef;
  my $bottomrule = undef;
  if ($_[0] =~ m/^[ \t\n\r]*\\hline/) {
    $toprule = '1px';
    $_[0]		= $';
  } elsif ($_[0] =~ m/^[ \t\n\r]*\\toprule/) {
    $_[0] = $';
    $toprule = '2px';
  }
  if ($_[0] =~ m/\\end{tabular}/ ) {
    $_[0] = $';
    local $_ = $`;
    if (m/\\hline[ \t\r\n]*$/) {
      $_ = $`;
      $bottomrule = '1px';
    } elsif (m/\\bottomrule[ \t\r\n]*$/) {
      $_ = $`;
      $bottomrule = '2px';
    }

    s|&|</td><td>|g;
    s|\\cr\\hline|</td></tr><tr style="border-top:1px solid gray;"><td>|g;
    s|\\cr\\midrule|</td></tr><tr><td colspan="999" style="margin:0pt;padding:0pt;border:0pt;height:1px;background:gray;"></td></tr><tr><td>|g;
    s|\\cr\\bottomrule|</td></tr><tr style="border-top:2px solid gray;"><td>|g;
    s|\\cr|</td></tr><tr><td>|g;
    s|\\\\|</td></tr><tr><td>|g;
    s|<tr><td>[ \t\r\n]*</td></tr></table>$|</table>|;
    $_[0] = $_."</td></tr></table>".$_[0];
    if (defined $toprule) {
      if (defined $bottomrule) {
	$_ = " style=\"border-top:$toprule solid gray;border-bottom:$bottomrule solid gray;\"";
      } else {
	$_ = " style=\"border-top:$toprule solid gray;\"";
      }
    } elsif (defined $bottomrule) {
      $_ = " style=\"border-top:$bottomrule solid gray;\"";
    } else {
      $_ = '';
    }
    return "<table$_ cellpadding=\"1\" cellspacing=\"1\"><tr><td>";
  }
  $_[TABULAR_COUNT]  = 1;
  $_[TABULAR_FORMAT] = $fmt;
  return "<table><tr><td>";
}

=pod

---++ %LaTeXSyntax

This hash contains most information about which and how to translate
the input syntax to HTML.

=cut

my %LaTeXSyntax	      = (
  '!' 		      => sub { # Special treatment of ! as TWiki quotation symbol
    if ($_[0] =~ m/^\\([a-zA-Z]+|.)/) {
      $_[0] = $';
      return $&;
    } else {
      return '!';
    }
  },
  '<' 		      => sub { # Pass XML tags through
    if ($_[0] =~ m/>/) {
      $_[0] = $';
      return '<' . $` . $&;
    }
    return '<';
  },
  '-' 		      => sub {
    return '-';
  },
  '~' 		      => sub {
    return '~';
  },
  '%' 		      => sub {
    return '%';
  },
  '^' 		      => sub {
    return '^' if $_[STATE] != STATE_MATH;
    return "<sup>".expanded_arg(@_)."</sup>";
  },
  '_' 		      => sub {
    return '_' if $_[STATE] != STATE_MATH;
    return "<sub>".expanded_arg(@_)."</sub>";
  },
  '\\include'         => \&include,
  '\\input'           => \&include,
  '\\verb'	      => \&macro_verb,
  '\\includegraphics' => \&macro_includegraphics,
  '\\begin{verbatim}' => \&env_verbatim,
  '\\begin{center}'   => '<center>',
  '\\end{center}'     => '</center>',
  '\\begin{quote}'    => '<blockquote>',
  '\\end{quote}'      => '</blockquote>',
  '\\begin{quotation}'=> '<blockquote>',
  '\\end{quotation}'  => '</blockquote>',
  '\\begin{table}'    => [1,0,'<div style="background:#eeeeee;">',
#			  TWiki::Func::getPreferencesValue("LATEXSYNTAXPLUGIN_TABLENAME") || 
			  'Table'],
  '\\end{table}'      => [0,0,'</div>',''],
  '\\begin{figure}'   => [1,0,'<div style="background:#eeeeee;">',
#			  TWiki::Func::getPreferencesValue("LATEXSYNTAXPLUGIN_FIGURENAME") || 
			  'Figure'],
  '\\end{figure}'     => [0,0,'</div>',''],
  '\\caption'         => sub {
    local $_ = scan_arg(@_) || '';
    return "<div style=\"font-size:90%;font-style:normal;font-weight:normal;\"><i>$_[FLOAT_TYPE]:</i> $_</div>";
  },
  '\\begin{tabular}'  => \&begin_tabular,
  '\\end{tabular}'    => sub {
    $_[TABULAR_COUNT]  = undef;
    return "</td></tr></table>";
  },
  '&' 		      => sub {
    return '&' if not defined $_[TABULAR_COUNT];
    $_[TABULAR_COUNT]++;
    return '</td><td>'
  },
  '\\\\' 	      => sub {
    return '<br />' if not defined $_[TABULAR_COUNT];
    $_[TABULAR_COUNT] = 0;
    return '</td></tr><tr><td>'
  },
  '\\cr' 	      => sub {
    return '' if not defined $_[TABULAR_COUNT];
    $_[TABULAR_COUNT] = 0;
    return '</td></tr><tr><td>'
  },
  '\\ '		         => ' ',
  '\\$'		         => '$',
  '\\%'		         => '&#37;',
  '\\_'		         => '_',
  '\\('		         => \&begin_math,
  '\\)'		         => \&end_math,
  '\\begin{math}'	 => \&begin_math,
  '\\end{math}'		 => \&end_math,
  '\\['			 => \&begin_displaymath,
  '\\]'			 => \&end_displaymath,
  '\\begin{displaymath}' => \&begin_displaymath,
  '\\end{displaymath}'	 => \&end_displaymath,
  '\\{'			 => '{',
  '\\}'			 => '}',
  '\\-'		         => '&shy;',
  '\\!'		         => '&thinsp;',
  '\\,'		         => ' ',
  '\\relax'	         => '',
  '\\smallskip'	         => '',
  '\\medskip'	         => '',
  '\\bigskip'	      => '',
  '\\ae'	      => '&aelig;',
  '\\AE'	      => '&AElig;',
  '\\oe'	      => '&oelig;',
  '\\OE'	      => '&OElig;',
  '\\o'		      => '&oslash;',
  '\\O'		      => '&Oslash;',
  '\\aa'	      => '&aring;',
  '\\AA'	      => '&Aring;',
  '\\ss'	      => '&szlig;',
  '\\par'	      => '<p>',
  '\\noindent'	      => '',
  '\\TeX'	      => 'T<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',
  '\\eTeX'	      => '&epsilon;-<span style="margin-left:-0.15em;">T</span><span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',
  '\\ExTeX'	      => '&epsilon;&chi;<span style="margin-left:-0.15em;">T</span><span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',
  '\\LaTeX'	      => 'L<span style="text-transform:uppercase;font-size:75%;vertical-align:0.45ex;margin-left: -0.36em;margin-right: -0.08em;">a</span>T<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',,
  '\\LaTeXe'	      => 'L<span style="text-transform:uppercase;font-size:75%;vertical-align:0.45ex;margin-left: -0.36em;margin-right: -0.08em;">a</span>T<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X2&epsilon;',
  '\\BibTeX'	      => '<span style="font-variant:small-caps;">Bib</span>T<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',
  '\\XeTeX'	      => 'X<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.07em;line-height:0;">&#398;</span>T<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>X',
  '\\ConTeXt'	      => '!ConT<span style="text-transform:uppercase;font-size:95%;vertical-align:-0.45ex;margin-left:-0.1em;margin-right:-0.06em;line-height:0;">e</span>Xt',
  '\\label'	      => [0,1,'<a name="#1"></a>'],
  '\\ref'	      => [0,1,'<a href="#1">???</a>'],
  '\\pageref'	      => [0,1,'<a href="#1">1</a>'],
  '\\href'	      => [0,2,'<a href="#1">#2</a>'],
  '\\Include'	      => [0,2,'<a href="#1">#2</a>'],
  '\\url'	      => [0,1,'<a href="#1">#1</a>'],
  '\\section'	      => [1,1,'<h1>#1</h1>'],
  '\\subsection'      => [1,1,'<h2>#1</h2>'],
  '\\subsubsection'   => [1,1,'<h3>#1</h3>'],
  '\\paragraph'	      => [1,1,'<h4>#1</h4>'],
  '\\subparagraph'    => [1,1,'<h5>#1</h5>'],
  '\\subsubparagraph' => [1,1,'<h6>#1</h6>'],
  '\\emph'	      => [0,1,'<em>#1</em>'],
  '\\textbf'	      => [0,1,'<b>#1</b>'],
  '\\textit'	      => [0,1,'<i>#1</i>'],
  '\\texttt'	      => [0,1,'<tt>#1</tt>'],
  '\\textsf'	      => [0,1,'<span style="font-family:sans-serif;">#1</span>'],
  '\\textrm'	      => [0,1,'<span style="font-family:serif;">#1</span>'],
  '\\textsc'	      => [0,1,'<span style="font-variant:small-caps;">#1</span>'],
  '\\footnote'	      => sub {
    local $_	      = expanded_arg(@_);
    my $fn	      = ++$_[FOOTNOTE_COUNT];
    return "<sup>$fn</sup><div style=\"float:right;width:25%;font-size:80%;background:#eeeeff;padding:.5em;border:1px solid gray;\"><sup>$fn</sup> $_</div>";
  },
  '\\marginpar'  => sub {
    scan_opt(@_);
    local $_ = expanded_arg(@_);
    return "<span style=\"float:right;width:25%;font-size:80%;background:#eeeeff;padding:.5em;border:1px solid gray;\">$_</span>";
  },
  '\\begin{itemize}'  => sub {
    push @{$_[STATE_STACK]}, $_[STATE];
    $_[STATE] = STATE_ITEMIZE;
    return '<ul>';
  },
  '\\end{itemize}'    => sub {
    $_[STATE] = pop @{$_[STATE_STACK]};
    return '</ul>';
  },
  '\\begin{enumerate}'  => sub {
    push @{$_[STATE_STACK]}, $_[STATE];
    $_[STATE] = STATE_ENUMERATE;
    return '<ol>';
  },
  '\\end{enumerate}'    => sub {
    $_[STATE] = pop @{$_[STATE_STACK]};
    return '</ol>';
  },
  '\\begin{description}' => sub {
    push @{$_[STATE_STACK]}, $_[STATE];
    $_[STATE] = STATE_DESCRIPTION_1;
    return '<dl>';
  },
  '\\end{description}' => sub {
    $_[STATE] = pop @{$_[STATE_STACK]};
    return '</dd></dl>';
  },
  '\\item'             => sub {
    local $_ = scan_opt(@_);
    if ($_[STATE] == STATE_DESCRIPTION_1) {
      $_[STATE] = STATE_DESCRIPTION;
      return "<dt>$_</dt><dd>";
    } elsif ($_[STATE] == STATE_DESCRIPTION) {
      return "</dd><dt>$_</dt><dd>";
    } else {
      return '<li>';
    }
  },
  '\\day'             => sub {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
	localtime(time);
    return $mday;
  },
  '\\month'             => sub {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
	localtime(time);
    return $mon + 1;
  },
  '\\year'             => sub {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
	localtime(time);
    return $year + 1900;
  },
  '\\today'             => sub {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
	localtime(time);
    $year  += 1900;
    my $monthNames = TWiki::Func::getPreferencesValue("LATEXSYNTAXPLUGIN_MONTH")
    || "January February March April May June July Augaugust September October November December"; 
    my @monthNames = split(/ /,$monthNames);
    my $month	   = $monthNames[$mon];
    local $_       = TWiki::Func::getPreferencesValue("LATEXSYNTAXPLUGIN_DATEFORMAT") || "MMM D, YYYY"; 
    s/MMM/$month/g;
    s/M/$mon/g;
    s/YYYY/$year/g;
    my $y   = sprintf("%2d",$year % 1000);
    s/YY/$y/g;
    my $d   = sprintf("%2d",$mday);
    s/DD/$d/g;
    s/D/$mday/g;
    return "$month $mday, $year";
  },
  '\\hbox'            => [0,1,'#1'],
  '\\begin{minipage}' => [1,1,'<span style="width:#1">'],
  '\\end{minipage}'   => [0,0,'</span>'],
  '\\parbox'          => [1,2,'<span style="width:#1">#2</span>'],

  '\\thispagestyle'   => [0,1,''],
  '\\pagestyle'       => [0,1,''],
  '\\enlargethispage' => [0,1,''],
  '\\pagebreak'       => [1,0,''],
  '\\nopagebreak'     => [1,0,''],
  '\\newpage'         => '',
  '\\clearpage'       => '',
  '\\cleardoublepage' => '',
  '\\pagebreak'       => '',
  '\\fussy'           => '',
  '\\sloppy'          => '',
  '\\/'               => '',
  '\\index'           => [0,1,''],
  '\\cite'            => [1,0,''],
  '\\nocite'          => '',

  '\\symbol'          => [0,1,'&##1;'],

  '\\iftex'           => sub {
    if ($_[0] =~ m/\\fi/) {
      $_[0] = $';
    } else {
      $_[0] = '';
    }
    return '';
  },
  '\\ifhtml'          => sub {
    if ($_[0] =~ m/\\fi/) {
      $_[0] = $';
      return $`;
    } else {
      local $_ = $_[0];
      $_[0]    = '';
      return $_;
    }
  },

  '\\"'		      => +{'a'   => '&auml;',
		          'o'   => '&ouml;',
		          'u'   => '&uuml;',
		          'e'   => '&euml;',
		          'i'   => '&iuml;',
		          '\\i' => '&iuml;',
		          'A'   => '&Auml;',
		          'O'   => '&Ouml;',
		          'U'   => '&Uuml;',
		          'E'   => '&Euml;',
		          'I'   => '&Iuml;',
		          '\\I' => '&Iuml;',
		          'y'   => '&yuml;',
		          ' '   => '&uml;',
		          ''    => '&uml;',
		         },
  '\\^'		      => +{'a'   => '&acirc;',
		          'o'   => '&ocirc;',
		          'u'   => '&ucirc;',
		          'e'   => '&ecirc;',
		          'i'   => '&icirc;',
		          '\\i' => '&icirc;',
		          'A'   => '&Acirc;',
		          'O'   => '&Ocirc;',
		          'U'   => '&Ucirc;',
		          'E'   => '&Ecirc;',
		          'I'   => '&Icirc;',
		          '\\I' => '&Icirc;',
		         },
  '\\\''	      => +{'a'   => '&aacute;',
		          'o'   => '&oacute;',
		          'u'   => '&uacute;',
		          'e'   => '&eacute;',
		          'i'   => '&iacute;',
		          '\\i' => '&iacute;',
		          'A'   => '&Aacute;',
		          'O'   => '&Oacute;',
		          'U'   => '&Uacute;',
		          'E'   => '&Eacute;',
		          'I'   => '&Iacute;',
		          '\\I' => '&Iacute;',
		          ''    => '&acute;',
		          ' '   => '&acute;',
		         },
  '\\`'		      => +{'a'   => '&agrave;',
		          'o'   => '&ograve;',
		          'u'   => '&ugrave;',
		          'e'   => '&egrave;',
		          'i'   => '&igrave;',
		          '\\i' => '&igrave;',
		          'A'   => '&Agrave;',
		          'O'   => '&Ograve;',
		          'U'   => '&Ugrave;',
		          'E'   => '&Egrave;',
		          'I'   => '&Igrave;',
		          '\\I' => '&Igrave;',
		          ''    => '&grave;',
		          ' '   => '&grave;',
		         },
  '\\c'		      => {'c' => '&ccedil;',
		          'C' => '&Ccedil;',
		          ' ' => '&cedil;',
		          ''  => '&cedil;',
		         },
  '\\~'		      => {'a' => '&atilde;',
			  'o' => '&otilde;',
			  'n' => '&ntilde;',
			  'A' => '&Atilde;',
			  'O' => '&Otilde;',
			  'N' => '&Ntilde;',
			  ''  => '~',
			  ' ' => '~',
		         },
  '\\not'	      => {'=' => '&ne;',
			  '\\in' => '&notin;',
			  '\\subset' => '&nsub;',
		         },

  '\\P'			=> '&para;',
  '\\S'			=> '',
  '\\dots'		=> '&hellip;',
  '\\ldots'		=> '&hellip;',

  '\\euro'		=> '&euro;',
  '\\texteuro'		=> '&euro;',
  '\\textcent'		=> '&cent;',
  '\\textdollar'	=> '$',
  '\\textyen'		=> '&yen;',
  '\\texttrademark'	=> '&trade;',
  '\\textcopyright'	=> '&copy;',
  '\\textregistered'	=> '&reg;',
  '\\textcelsius'	=> '&deg;',
  '\\textperthousand'	=> '&permil;',
  '\\textonesuperior'	=> '&sup1;',
  '\\texttwosuperior'	=> '&sup2;',
  '\\textthreesuperior'	=> '&sup3;',
  '\\textonequater'	=> '&frac14;',
  '\\textonehalf'	=> '&frac12;',
  '\\textthreequaters'	=> '&frac34;',
  '\\textdiv'		=> '&divide;',
  '\\textmu'		=> '&micro;',

  '\\alpha'		  => '&alpha;',
  '\\theta'		  => '&theta;',
  '\\tau'		  => '&tau;',
  '\\beta'		  => '&beta;',
#  '\\vartheta'		  => '',
  '\\pi'		  => '&pi;',
  '\\upsilon'		  => '&upsilon;',
  '\\gamma'		  => '&gamma;',
  '\\varpi'		  => '&piv;',
  '\\phi'		  => '&phi;',
  '\\delta'		  => '&delta;',
  '\\kappa'		  => '&kappa;',
  '\\rho'		  => '&rho;',
#  '\\varphi'		  => '',
  '\\epsilon'		  => '&epsilon;',
  '\\lambda'		  => '&lambda;',
#  '\\varrho'		  => '',
  '\\chi'		  => '&chi;',
  '\\varepsilon'	  => '&epsilon;',
  '\\mu'		  => '&mu;',
  '\\sigma'		  => '&sigma;',
  '\\psi'		  => '&psi;',
  '\\zeta'		  => '&zeta;',
  '\\nu'		  => '&nu;',
  '\\varsigma'		  => '&sigmaf;',
  '\\omega'		  => '&omega;',
  '\\eta'		  => '&eta;',
  '\\xi'		  => '&xi;',
  '\\Gamma'		  => '&Gamma;',
  '\\Lambda'		  => '&Lambda;',
  '\\Sigma'		  => '&Sigma;',
  '\\Psi'		  => '&Psi;',
  '\\Delta'		  => '&Delta;',
  '\\Xi'		  => '&Xi;',
  '\\Upsilon'		  => '&Upsilon;',
  '\\Omega'		  => '&Omega;',
  '\\Theta'		  => '&Theta;',
  '\\Pi'		  => '&Pi;',
  '\\Phi'		  => '&Phi;',
  '\\pm'		  => '&plusmn;',
  '\\cap'		  => '&cap;',
  '\\diamond'		  => '&diams;',
  '\\oplus'		  => '&oplus;',
#  '\\mp'		  => '&#x2213;',
  '\\cup'		  => '&cup;',
#  '\\bigtriangleup'	  => '',
#  '\\ominus'		  => '',
  '\\times'		  => '&times;',
#  '\\uplus'		  => '',
#  '\\bigtriangledown'	  => '',
  '\\otimes'		  => '&otimes;',
  '\\div'		  => '&divide;',
#  '\\sqcap'		  => '',
#  '\\triangleleft'	  => '',
#  '\\oslash'		  => '',
  '\\ast'		  => '&lowast;',
#  '\\sqcup'		  => '',
#  '\\triangleright'	  => '',
#  '\\odot'		  => '',
  '\\star'		  => '*',
  '\\vee'		  => '&or;',
#  '\\lhd'		  => '',
#  '\\bigcirc'		  => '',
#  '\\circ'		  => '',
  '\\wedge'		  => '&and;',
#  '\\rhd'		  => '',
  '\\dagger'		  => '&dagger;',
  '\\bullet'		  => '&bull;',
  '\\setminus'		  => '\\',
#  '\\unlhd'		  => '',
  '\\ddagger'		  => '&Dagger;',
  '\\cdot'		  => '&middot;',
#  '\\wr'		  => '',
#  '\\unrhd'		  => '',
#  '\\amalg'		  => '',
  '\\leq'		  => '&le;',
  '\\geq'		  => '&ge;',
  '\\equiv'		  => '&equiv;',
#  '\\models'		  => '',
#  '\\prec'		  => '',
#  '\\succ'		  => '',
  '\\sim'		  => '&sim;',
  '\\perp'		  => '&bot;',
#  '\\preceq'		  => '',
#  '\\succeq'		  => '',
  '\\simeq'		  => '&cong;',
  '\\mid'		  => '|',
  '\\ll'		  => '&lt;<span style="margin-left:-.3em;">&lt;</span>',
  '\\gg'		  => '&gt;<span style="margin-left:-.3em;">&gt;</span>',
#  '\\asymp'		  => '',
#  '\\parallel'		  => '',
  '\\subset'		  => '&sub;',
  '\\supset'		  => '&sup;',
  '\\approx'		  => '&asymp;',
#  '\\bowtie'		  => '',
  '\\subseteq'		  => '&sube;',
  '\\supseteq'		  => '&supe;',
#  '\\cong'		  => '',
#  '\\Join'		  => '',
#  '\\sqsubset'		  => '',
#  '\\sqsupset'		  => '',
  '\\neq'		  => '&ne;',
#  '\\smile'		  => '',
#  '\\sqsubseteq'	  => '',
#  '\\sqsupseteq'	  => '',
#  '\\doteq'		  => '',
#  '\\frown'		  => '',
  '\\in'		  => '&isin;',
  '\\ni'		  => '&ni;',
  '\\propto'		  => '&prop;',
#  '\\vdash'		  => '',
#  '\\dashv'		  => '',
  '\\colon'		  => ':',
#  '\\ldotp'		  => '',
#  '\\cdotp'		  => '',
  '\\leftarrow'		  => '&larr;',
#  '\\longleftarrow'	  => '',
  '\\uparrow'		  => '&uarr;',
#  '\\Leftarrow'	  => '',
#  '\\Longleftarrow'	  => '',
  '\\Uparrow'		  => '&uArr;',
  '\\rightarrow'	  => '&rarr;',
#  '\\longrightarrow'	  => '',
  '\\downarrow'		  => '&darr;',
  '\\Rightarrow'	  => '&rArr;',
#  '\\Longrightarrow'	  => '',
  '\\Downarrow'		  => '&dArr;',
  '\\leftrightarrow'	  => '&harr;',
#  '\\longleftrightarrow' => '',
#  '\\updownarrow'	  => '',
  '\\Leftrightarrow'	  => '&hArr;',
#  '\\Longleftrightarrow' => '',
#  '\\Updownarrow'	  => '',
#  '\\mapsto'		  => '',
#  '\\longmapsto'	  => '',
#  '\\nearrow'		  => '',
#  '\\hookleftarrow'	  => '',
#  '\\hookrightarrow'	  => '',
#  '\\searrow'		  => '',
#  '\\leftharpoonup'	  => '',
#  '\\rightharpoonup'	  => '',
#  '\\swarrow'		  => '',
#  '\\leftharpoondown'	  => '',
#  '\\rightharpoondown'	  => '',
#  '\\nwarrow'		  => '',
#  '\\rightleftharpoons'  => '',
#  '\\leadsto'		  => '',
#  '\\ldots'		  => '',
#  '\\cdots'		  => '',
#  '\\vdots'		  => '',
#  '\\ddots'		  => '',
  '\\aleph'		  => '&alefsym;',
  '\\prime'		  => '\'',
  '\\forall'		  => '&forall;',
  '\\infty'		  => '&infin;',
#  '\\hbar'		  => '',
  '\\emptyset'		  => '&empty;',
  '\\exists'		  => '&exist;',
#  '\\Box'		  => '',
#  '\\imath'		  => '',
  '\\nabla'		  => '&nabla;',
  '\\neg'		  => '&not;',
#  '\\Diamond'		  => '',
#  '\\jmath'		  => '',
#  '\\surd'		  => '',
#  '\\flat'		  => '',
#  '\\triangle'		  => '',
#  '\\ell'		  => '',
#  '\\top'		  => '',
#  '\\natural'		  => '',
  '\\clubsuit'		  => '&clubs;',
  '\\wp'		  => '&weierp;',
  '\\bot'		  => '&perp;',
#  '\\sharp'		  => '',
  '\\diamondsuit'	  => '&diams;',
  '\\Re'		  => '&real;',
  '\\backslash'		  => '\\',
  '\\heartsuit'		  => '&hearts;',
  '\\Im'		  => '&image;',
  '\\angle'		  => '&ang;',
  '\\partial'		  => '&part;',
  '\\spadesuit'		  => '&spades;',
#  '\\mho'		  => '',
  '\\sum'		  => '&sum;',
#  '\\bigcap'		  => '',
#  '\\bigodot'		  => '',
  '\\prod'		  => '&prod;',
#  '\\bigcup'		  => '',
#  '\\bigotimes'	  => '',
#  '\\coprod'		  => '',
#  '\\bigsqcup'		  => '',
#  '\\bigoplus'		  => '',
  '\\int'		  => '&int;',
#  '\\bigvee'		  => '',
#  '\\biguplus'		  => '',
#  '\\oint'		  => '',
#  '\\bigwedge'		  => '',
  '\\arccos'		  => '<span style="font-style:normal;">arccos</span>',
  '\\cos'		  => '<span style="font-style:normal;">cos</span>',
  '\\csc'		  => '<span style="font-style:normal;">csc</span>',
  '\\exp'		  => '<span style="font-style:normal;">exp</span>',
  '\\ker'		  => '<span style="font-style:normal;">ker</span>',
  '\\limsup'		  => '<span style="font-style:normal;">limsup</span>',
  '\\min'		  => '<span style="font-style:normal;">min</span>',
  '\\sinh'		  => '<span style="font-style:normal;">sinh</span>',
  '\\arcsin'		  => '<span style="font-style:normal;">arcsin</span>',
  '\\cosh'		  => '<span style="font-style:normal;">cosh</span>',
  '\\deg'		  => '<span style="font-style:normal;">deg</span>',
  '\\gcd'		  => '<span style="font-style:normal;">gcd</span>',
  '\\lg'		  => '<span style="font-style:normal;">lg</span>',
  '\\ln'		  => '<span style="font-style:normal;">ln</span>',
  '\\Pr'		  => '<span style="font-style:normal;">Pr</span>',
  '\\sup'		  => '<span style="font-style:normal;">sup</span>',
  '\\arctan'		  => '<span style="font-style:normal;">arctan</span>',
  '\\cot'		  => '<span style="font-style:normal;">cot</span>',
  '\\det'		  => '<span style="font-style:normal;">det</span>',
  '\\hom'		  => '<span style="font-style:normal;">hom</span>',
  '\\lim'		  => '<span style="font-style:normal;">lim</span>',
  '\\log'		  => '<span style="font-style:normal;">log</span>',
  '\\sec'		  => '<span style="font-style:normal;">sec</span>',
  '\\tan'		  => '<span style="font-style:normal;">tan</span>',
  '\\arg'		  => '<span style="font-style:normal;">arg</span>',
  '\\coth'		  => '<span style="font-style:normal;">coth</span>',
  '\\dim'		  => '<span style="font-style:normal;">dim</span>',
  '\\inf'		  => '<span style="font-style:normal;">inf</span>',
  '\\liminf'		  => '<span style="font-style:normal;">liminf</span>',
  '\\max'		  => '<span style="font-style:normal;">max</span>',
  '\\sin'		  => '<span style="font-style:normal;">sin</span>',
  '\\tanh'		  => '<span style="font-style:normal;">tanh</span>',
  '\\uparrow'		  => '&uarr;',
  '\\Uparrow'		  => '&uArr;',
  '\\downarrow'		  => '&darr;',
  '\\Downarrow'		  => '&dArr;',
#  '\\updownarrow'	  => '',
#  '\\Updownarrow'	  => '',
  '\\lfloor'		  => '&lfloor;',
  '\\rfloor'		  => '&rfloor;',
  '\\lceil'		  => '&lceil;',
  '\\rceil'		  => '&rceil;',
  '\\langle'		  => '&lang;',
  '\\rangle'		  => '&rang;',
  '\\backslash'		  => '\\',
#  '\\rmoustache'	  => '',
#  '\\lmoustache'	  => '',
#  '\\rgroup'		  => '',
#  '\\lgroup'		  => '',
#  '\\arrowvert'	  => '',
#  '\\Arrowvert'	  => '',
#  '\\bracevert'	  => '',

   '\\'			  => '' # DO NOT DELETE
);

=pod

---++ expand_LaTeX($input)

Expand macros in the input string.

=cut

no strict 'refs';
sub expand_LaTeX {
  my $result = '';
  local $_;

  while ($_[0] =~ m/[-!<_^\\&]/) {
    $result .= $`;
    $_[0] = $';
    my $key = $&;
    if ($& eq '\\' &&
	($_[0] =~ m/^(begin{[a-zA-Z]*})[ \t]*/
	 ||$_[0] =~ m/^(end{[a-zA-Z]*})[ \t]*/
	 || $_[0] =~ m/^([a-zA-Z]+|.)[ \t]*/)) {
      $_[0] = $';
      $key .= $1;
    }

    $_   = $LaTeXSyntax{$key};

    if (not defined($_)) {
      TWiki::Func::writeWarning("$key not defined");
      $result .= "<span style='color:red;background:pink;text-decoration:blink;'>?$key?</span>";

    } elsif (ref($_) eq 'CODE') {
      TWiki::Func::writeWarning("$key is code") if $debug;
      $result .= &$_(@_);

    } elsif (ref($_) eq 'ARRAY') {
      TWiki::Func::writeWarning("$key has args") if $debug;
      my ($opt, $args, $text, $float) = @$_;
      if ($opt > 0) {
	$_ = scan_opt(@_) || '';
	my $save = $_[0];
	$_[0]	 = $_;
	$_	 = expand_LaTeX(@_);
	$_[0]	 = $save;
	$text =~ s/#0/$_/g;
      }
      for(my $i = 1; $i <= $args; $i++) {
	$_    = expanded_arg(@_);
	$text =~ s/#$i/$_/g;
      }
      $result .= $text;
      $_[FLOAT_TYPE] = $float if defined $float;

    } elsif (ref($_) eq 'HASH') {
      TWiki::Func::writeWarning("$key is hashed") if $debug;
      my $a = scan_arg(@_) || '';
      $_ = $$_{$a};
      if (defined($_)) {
	$result .= $_;
      } else {
	TWiki::Func::writeWarning("mapping of $a is undefined");
      }

    } else {
      TWiki::Func::writeWarning("$key is mapped") if $debug;
      $result .= $_;
    }
  }
  $_[0] = $result . $_[0];
}

=pod

---++ preRenderingHandler( $text, \%map )
   * =$text= - text, with the head, verbatim and pre blocks replaced with placeholders
   * =\%removed= - reference to a hash that maps the placeholders to the removed blocks.

Handler called immediately before TWiki syntax structures (such as lists) are
processed, but after all variables have been expanded. Use this handler to 
process special syntax only recognised by your plugin.

=cut
sub preRenderingHandler {
  #my( $text, $pMap ) = @_;

  return if $TWiki::cfg{LaTeXSyntaxPlugin}{Disabled};

  $_[STATE] = STATE_TEXT;
  $_[STATE_STACK] = [];
  expand_LaTeX(@_);
}

1;
